% K2-SURF model for limonene adsorption on silica particles
% This code makes Figure 5 in Fang et al. Chem. Sci. 2019, 10, 2906

%Species: 1.Adsorbed C* down limonene configuration, 2. Gas-phase limonene, 
% 3. Adsorbed C* up limonene configuration

clear all
warning off all
close all
global sigma a0 W kd ksr ratio_down


%**********constants******%
NA=6.02e23; % Avogadro's number (mol-1)
R=8.31e7; % Gas constant (g cm2 s-2 K-1 mol-1)
T=273+23; % Temperature (K)
R_H=82.05746; % Gas constant (cm3 atm K-1 mol-1)

%***********Parameters for limonene**********
%---------Limonene parameters-----------------

Dens(1)=0.841; % Density of limonene (g cm-3)
Mx(1)=136.24; % Molar mass of limonene (g mol-1)
V_w(1)=Mx(1)/NA/Dens(1); % Volume of one limonene molecule (cm3)
delta_x(1)=(V_w(1))^(1/3); % Geometrical diameter of limonene (cm)
sigma(1)=5.49e-15; % Effective molecular cross section of limonene (cm2)
W(1)=sqrt(8*R*T/pi/Mx(1)); % Mean thermal velocity of limonene (cm s-1)

%-------------Initial parameters---------------

Exp_number=[1,2,3]; %Three different experiments are modeled
for MM=1:3
    Exp_number_1=Exp_number(MM);
        
Td(1)=22622.4e-9; % Desorption lifetime of the C* down limonene configuration on the SiO2 surface (s)
Td(2)=22622.4e-9; % Desorption lifetime of the C* up limonene configuration on the SiO2 surface (s)
kd(1)=1/Td(1); % Desorption rate of the C* down limonene configuration from the SiO2 surface (s-1)
kd(2)=1/Td(2); % Desorption rate of the C* up limonene configuration from the SiO2 surface (s-1)
a0=1; %Surface mass accommodation of limonene on an adsorbate free surface (no units)
ka=(a0*W(1))/4; % Adsorption rate of limonene on an adsorbate free surface (cm s-1)
ksr(1)=4.6e9; % First order rate coefficient for the conversion of the C* down limonene configuration to the the C* up limonene configuration (s-1)
ksr(2)=6.8e8; % First order rate coefficient for the conversion of the C* up limonene configuration to the the C* down limonene configuration (s-1)
ratio_down=0.5; % Fraction of limonene adsorbed as C* down
%------------------------------------------------------
for NN=1:1980 % The experiment last for 1980 seconds and we constrain to the pressure and solve differential equations after every second.
    if NN==1 %First time point (1 second)
%-----------Initial gas-phase concentrations---------------
% Here we constrain to the measured pressures after 1 second (in mTorr) and convert to gas-phase concentrations

if MM==1 %experiment 1
y0(2)=(1.1323E-16*(NN)^6-3.9201E-13*(NN)^5+5.0060E-10*(NN)^4-2.6751E-07*(NN)^3+...
    2.7604E-05*(NN)^2+2.5125E-02*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3 experiment 1)
elseif MM==2 %experiment 1
y0(2)=(1.0831E-16*(NN)^6-3.4382E-13*(NN)^5+3.5828E-10*(NN)^4-7.2139E-08*(NN)^3-...
    1.1143E-04*(NN)^2+7.4851E-02*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3 experiment 2)
elseif MM==3 %experiment 1
y0(2)=(-8.0429E-17*(NN)^6+3.6551E-13*(NN)^5-6.9120E-10*(NN)^4+7.0924E-07*(NN)^3-...
    4.2636E-04*(NN)^2+1.4515E-01*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3 experiment 3)
end
y0(1)=0;% Initial concentration of the adsorbed C* down limonene configuration (molec. cm-2)
y0(3)=0;% Initial concentration of the adsorbed C* up limonene configuration (molec. cm-2)

%----------Solve differential equations-----------
n=500; %outputted time steps
t=logspace(-3,log10(1),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@K2SURF_limonene_F,t,y0); %solver

%Outputs
for i=1:n
    Time_output(NN,1)=NN; %Time (s)
    Gas_phase_pressure(NN,MM)=y0(2)/1e-3*760*(R_H*T)/6.02e23; %Gas-phase pressure (mTorr)
    Surf_conc_output(NN,MM)=Y(n,1)+Y(n,3); %Total adsorbed limonene (molec. cm-2)
end

    else
%-----------Initial concentrations---------------
% Here we constrain to the measured pressures at different time points (in mTorr) and convert to gas-phase concentrations

if MM==1 %Experiment 1
if NN<=1080
y0(2)=(1.1323E-16*(NN)^6-3.9201E-13*(NN)^5+5.0060E-10*(NN)^4-2.6751E-07*(NN)^3+...
    2.7604E-05*(NN)^2+2.5125E-02*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
else
y0(2)=(5.8651E-16*(NN-1080)^6-1.7430E-12*(NN-1080)^5+2.0327E-09*(NN-1080)^4-...
    1.1898E-06*(NN-1080)^3+3.7820E-04*(NN-1080)^2-6.9007E-02*(NN-1080)+7.0700E+00)*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
if y0(2)<=0
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
if NN>=1959
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
end
end

if MM==2 %Experiment 2
if NN<=1080
y0(2)=(1.0831E-16*(NN)^6-3.4382E-13*(NN)^5+3.5828E-10*(NN)^4-7.2139E-08*(NN)^3-...
    1.1143E-04*(NN)^2+7.4851E-02*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
else
y0(2)=(1.5679E-16*(NN-1080)^6-7.6655E-13*(NN-1080)^5+1.3802E-09*(NN-1080)^4-...
    1.2072E-06*(NN-1080)^3+5.5304E-04*(NN-1080)^2-1.3147E-01*(NN-1080)+ 1.4100E+01)*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
if y0(2)<=0
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
if NN>=1959
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
end
end

if MM==3 %Experiment 3
if NN<=1080
y0(2)=(-8.0429E-17*(NN)^6+3.6551E-13*(NN)^5-6.9120E-10*(NN)^4+7.0924E-07*(NN)^3-...
    4.2636E-04*(NN)^2+1.4515E-01*(NN))*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
else
y0(2)=(1.3942E-15*(NN-1080)^6-4.3473E-12*(NN-1080)^5+5.4010E-09*(NN-1080)^4-...
    3.4297E-06*(NN-1080)^3+1.1961E-03*(NN-1080)^2-2.3084E-01*(NN-1080)+2.1920E+01)*1e-3/760/(R_H*T)*6.02e23; % gas-phase concentration (molec. cm-3)
end
if y0(2)<=0
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
if NN>=1959
    y0(2)=0; % gas-phase concentration (molec. cm-3)
end
end

y0(1)=Y(n,1);% Concentration of the adsorbed C* down limonene configuration (molec. cm-2)
y0(3)=Y(n,3);% Concentration of the adsorbed C* up limonene configuration (molec. cm-2)

%----------Solve differential equations-----------
n=500; %outputted time steps
t=logspace(-3,log10(1),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@K2SURF_limonene_F,t,y0); %solver

%Outputs
for i=1:n
    Time_output(NN,1)=NN; %Time (s)
    Gas_phase_pressure(NN,MM)=y0(2)/1e-3*760*(R_H*T)/6.02e23; %Gas-phase pressure (mTorr)
    Surf_conc_output(NN,MM)=Y(n,1)+Y(n,3); %Total adsorbed limonene (molec. cm-2)
end

    end
end
%------------------Experimental measurements----------------------------

Experimental_data=xlsread('Measurements');
if Exp_number_1==1
    %Time (s)
    Time_exp=Experimental_data(:,1);
    Time_exp_2=Experimental_data(:,3);
    %Adorbed limonene (molec. cm-2) 
    Exp_conc=Experimental_data(:,2);
    %Pressure (mTorr)
    Exp_pressure=Experimental_data(:,4);
end

if Exp_number_1==2
    %Time (s)
    Time_exp=Experimental_data(:,6);
    Time_exp_2=Experimental_data(:,8);
    %Adorbed limonene (molec. cm-2) 
    Exp_conc=Experimental_data(:,7);
    %Pressure (mTorr)
    Exp_pressure=Experimental_data(:,9);
end
if Exp_number_1==3
    %Time (s)
    Time_exp=Experimental_data(:,11);
    Time_exp_2=Experimental_data(:,13);
    %Adorbed limonene (molec. cm-2) 
    Exp_conc=Experimental_data(:,12);
    %Pressure (mTorr)
    Exp_pressure=Experimental_data(:,14);
end

%Figures which will be outputted
figure(1)
subplot(3,1,Exp_number_1)
scatter(Time_exp,Exp_conc,'x')
hold on
plot(Time_output,Surf_conc_output(:,Exp_number_1),'k')
hold off
set(findall(gca, 'Type', 'Line'),'LineWidth',2);
xlabel('Time (s)')
ylabel('Surface conc. (cm-2)')
if Exp_number_1==1
    title('Experiment 1, Pressure = 7 mTorr')
    ylim([0 3e13]);
elseif Exp_number_1==2
    title('Experiment 2, Pressure = 14 mTorr')
    ylim([0 5e13]);
elseif Exp_number_1==3
    title('Experiment 3, Pressure = 22 mTorr')
    ylim([0 7e13]);
end

figure(2)
subplot(3,1,Exp_number_1)
scatter(Time_exp_2,Exp_pressure,'x')
hold on
plot(Time_output,Gas_phase_pressure(:,Exp_number_1),'k')
hold off
set(findall(gca, 'Type', 'Line'),'LineWidth',2);
xlabel('Time (s)')
ylabel({'Reaction cell';'pressure (mTorr)'})
if Exp_number_1==1
    title('Experiment 1, Pressure = 7 mTorr')
    ylim([0 8]);
elseif Exp_number_1==2
    title('Experiment 2, Pressure = 14 mTorr')
    ylim([0 16]);
elseif Exp_number_1==3
    title('Experiment 3, Pressure = 22 mTorr')
    ylim([0 25]);
end
end


